#include "stableheaders.hpp"

#include "game.hpp"

#include "gameobjects.hpp"
#include "hud.hpp"
#include "level.hpp"
#include "midi.hpp"
#include "shape.hpp"
#include "video.hpp"

//////////////////////////////////////////////////////////////////////////
// BSS
//////////////////////////////////////////////////////////////////////////
#pragma bss_seg(".bss_game")

Game g_game;

//////////////////////////////////////////////////////////////////////////
// DATA
//////////////////////////////////////////////////////////////////////////
#pragma data_seg(".data_game")

#include "autogenerated/SongMenu.inl"
#include "autogenerated/SongIngame.inl"

//////////////////////////////////////////////////////////////////////////
// CODE
//////////////////////////////////////////////////////////////////////////
#pragma code_seg(".code_game")

////////////////////////////////////////////////////////////////////////////////
//
//	Game_Init()
//
////////////////////////////////////////////////////////////////////////////////
/**
	\brief

		Set up the game.

*/
void Game_Init()
{
	// prepare game object list
	g_firstGameObject.pNext = &g_lastGameObject;
	g_lastGameObject.pPrev = &g_firstGameObject;

	// prepare line drawing
	D3D_VERIFY(D3DXCreateLine(g_pD3DDevice, &g_pD3DXLine), "D3DXCreateLine() failed!");
	D3D_VERIFY(g_pD3DXLine->SetWidth(SHAPE_PENSIZE), "SetWidth() failed!");
	D3D_VERIFY(g_pD3DXLine->SetAntialias(TRUE), "SetAntialias() failed!");

	// start the first game state
	Game_SwitchToMenu();
}

////////////////////////////////////////////////////////////////////////////////
//
//	Game_RunIngame()
//
////////////////////////////////////////////////////////////////////////////////
/**
	\brief

		Game state handler for the ingame state.

	\return

		If *true* the game should quit.

*/
bool Game_RunIngame()
{
	// handle wave and enemies
	Level_SpawnEnemies();
	
	// run and render all game objects
	Level_RunAndRender();

	// update the hud
	HUD_Render();

	// game over?
	if (g_player.pRunCallback == Player_RunDead)
		// yes! switch back to the main menu
		Game_SwitchToMenu();

	// pressed "esc" to quit?
	return GetAsyncKeyState(VK_ESCAPE);
}

////////////////////////////////////////////////////////////////////////////////
//
//	Game_RunMenu()
//
////////////////////////////////////////////////////////////////////////////////
/**
	\brief

		Game state handler for the menu game state.

	\return

		If *true* the game should quit.

*/
bool Game_RunMenu()
{
	// update the hud
	HUD_Render();
	
	// draw "main menu"
	HUD_DrawText
		(
		"vector wars 4k\n"
		"\n"
		"by jetset entertainment\n"
		"\n"
		"use mouse and cursor keys\n"
		"press [space]\n"
		, DT_WORDBREAK | DT_NOCLIP | DT_CENTER | DT_VCENTER
		);

	// pressed "space" to start the game?
	if (GetAsyncKeyState(VK_SPACE))
		// yes! so switch game state
		Game_SwitchToIngame();

	// pressed "esc" to quit the game?
	return GetAsyncKeyState(VK_ESCAPE);
}

////////////////////////////////////////////////////////////////////////////////
//
//	Game_SwitchToIngame()
//
////////////////////////////////////////////////////////////////////////////////
/**
	\brief

		Initialize the ingame game state.

*/
void Game_SwitchToIngame()
{
	// set up the first wave
	Level_Init();
	
	// reset the player
	Player_Init();

	// set the new game state handler
	g_game.pRunCallback = Game_RunIngame;
	
	// start the ingame music
	Midi_StartSong(g_songIngame);
}

////////////////////////////////////////////////////////////////////////////////
//
//	Game_SwitchToMenu()
//
////////////////////////////////////////////////////////////////////////////////
/**
	\brief

		Initializes the menu game state.

*/
void Game_SwitchToMenu()
{
	// set the new game state handler
	g_game.pRunCallback = Game_RunMenu;
	
	// start the menu music
	Midi_StartSong(g_songMenu);
}
